# Maya export script
# Copyright (C) The Pixel Farm Ltd. 2010-2024

import pfpy
import math
from pfpy import Tracker, Camera, Group, Mesh, Cloud

def pfExportName():
	return 'Autodesk Maya 2026 Ascii (zxy camera)'
	
def pfExportExtension():
	return 'ma'
	
def pfExportSupport():
	return ('cameras', 'groups', 'solvedTrackers', 'mocap', 'meshes', 'textures', 'pointClouds', 'multiCamera')

def modulus(i, m):
	return (i-math.floor(i/m)*m)

def check360(x, last):
	# see if adding 360 gets us nearer to the last angle
	if math.fabs(x+360.0-last) < math.fabs(x-last):
		return x+360.0
	# see if subtracting 360 gets us nearer to the last angle
	elif math.fabs(x-360.0-last) < math.fabs(x-last):
		return x-360.0
	return x
	
def main():
	fobj = open(pfpy.getExportFilename(), 'w')
	
	# set desired coordinate system
	pfpy.setExportCoordinateSystem('right', 'y')
	
	firstFrame= 0
	lastFrame= 0
	frameRate= 30.0

	# get the first and last frames
	if (pfpy.getNumCameras() > 0):
		c= pfpy.getCameraRef(0)
		firstFrame= c.getInPoint()
		lastFrame= c.getOutPoint()
		frameRate= c.getFrameRate()
	
	# version	
	fobj.write('//Maya ASCII 4.5 scene\n\n\n')
	fobj.write('requires maya "2019";\n')
	
	# use cm as default unit or uncomment the line below to use the units defined in the preferences.
	# This is not enabled by default since some users have reported problems using
	# anything other than centimeters as the units. Use at your own risk.
	units= 'cm'
	# units= pfpy.getUnitsString()

	fobj.write('currentUnit -l ' + units + ' -a deg ')

	rates = { 'film':24.0, 'pal':25.0, 'ntsc':30.0, 'show':48.0, 'palf':50.0, 'ntscf':60.0,
		  '23.976fps':23.976, '29.97fps':29.97, '47.952fps':47.952, '59.94fps':59.94 }

	tval = 'ntsc'
	dif = 100000.0
	for r in rates:
		d = math.fabs(rates[r]-frameRate)
		if d < dif :
			dif = d
			tval = r
	fobj.write('-t ' + tval + ';\n')

	#if math.fabs(frameRate-23.98) < 0.0001:
	#	fobj.write('-t film')
	#elif math.fabs(frameRate-24.0) < 0.0001:
	#	fobj.write('-t film')
	#elif math.fabs(frameRate-25.0) < 0.0001:
	#	fobj.write('-t pal')
	#elif math.fabs(frameRate-48.0) < 0.0001:
	#	fobj.write('-t show')
	#elif math.fabs(frameRate-50.0) < 0.0001:
	#	fobj.write('-t palf')
	#elif math.fabs(frameRate-60.0) < 0.0001:
	#	fobj.write('-t ntscf')
	#elif math.fabs(frameRate-59.94) < 0.01:
	#	fobj.write('-t ntscf')
	#else:
	#	fobj.write('-t ntsc')
	#fobj.write(';\n')

	fobj.write('\ncreateNode transform -n "pf_data";\n')
	
	hasFirstCamera = False
	eulOrder = 'zxy'

	numCameras = pfpy.getNumCameras()
	if numCameras > 0:
                firstCamera = pfpy.getCameraRef(0)
                hasFirstCamera = True

	idx = 0
	
	while (idx < numCameras):
		c = pfpy.getCameraRef(idx)
		
		if c.getExport():
			inPoint = c.getInPoint()
			outPoint = c.getOutPoint()
			numFrames = outPoint - inPoint +1
		
			fobj.write('\ncreateNode transform -n "'+c.getName()+'" -p "pf_data";\n')
			fobj.write('\tsetAttr ".ro" 2;\n')
			
			fobj.write('createNode camera -n "'+c.getName()+'Shape" -p "'+c.getName()+'";\n')
			fobj.write('\tsetAttr -k off ".v";\n')
			fobj.write('\tsetAttr ".cap" -type "double2" '+'%g'%(c.getSensorWidth('mm')/25.4)+' '+'%g'%(c.getSensorHeight('mm')/25.4)+';\n')
			fobj.write('\tsetAttr ".ff" 3;\n')
			fobj.write('\tsetAttr ".ncp" 0.1;\n')
			fobj.write('\tsetAttr ".fcp" 10000.0;\n')
			fobj.write('\tsetAttr ".fd" 5;\n')
			fobj.write('\tsetAttr ".coi" 5;\n')
			fobj.write('\tsetAttr ".ow" 10;\n')
			fobj.write('\tsetAttr ".dr" yes;\n')
			
			# focal lengths
			fobj.write('\ncreateNode animCurveTU -n "'+c.getName()+'_focalLength";\n')
			fobj.write('\tsetAttr -s '+'%d'%numFrames+' ".ktv['+'%d'%inPoint+':'+'%d'%outPoint+']"')
			for i in range(inPoint, outPoint+1):
				fobj.write(' '+'%d'%i+' '+'%g'%c.getFocalLength(i, 'mm'))
			fobj.write(';\n')
						
			# x translation
			fobj.write('\ncreateNode animCurveTL -n "'+c.getName()+'_transX";\n')
			fobj.write('\tsetAttr -s '+'%d'%numFrames+' ".ktv['+'%d'%inPoint+':'+'%d'%outPoint+']"')
			for i in range(inPoint, outPoint+1):
				p = c.getTranslation(i)
				fobj.write(' '+'%d'%i+' '+'%f'%p[0])
			fobj.write(';\n')
						
			# y translation
			fobj.write('\ncreateNode animCurveTL -n "'+c.getName()+'_transY";\n')
			fobj.write('\tsetAttr -s '+'%d'%numFrames+' ".ktv['+'%d'%inPoint+':'+'%d'%outPoint+']"')
			for i in range(inPoint, outPoint+1):
				p = c.getTranslation(i)
				fobj.write(' '+'%d'%i+' '+'%f'%p[1])
			fobj.write(';\n')
						
			# z translation
			fobj.write('\ncreateNode animCurveTL -n "'+c.getName()+'_transZ";\n')
			fobj.write('\tsetAttr -s '+'%d'%numFrames+' ".ktv['+'%d'%inPoint+':'+'%d'%outPoint+']"')
			for i in range(inPoint, outPoint+1):
				p = c.getTranslation(i)
				fobj.write(' '+'%d'%i+' '+'%f'%p[2])
			fobj.write(';\n')
						
			# x rotation
			fobj.write('\ncreateNode animCurveTA -n "'+c.getName()+'_rotX";\n')
			fobj.write('\tsetAttr -s '+'%d'%numFrames+' ".ktv['+'%d'%inPoint+':'+'%d'%outPoint+']"')
			
			last = 0.0
			for i in range(inPoint, outPoint+1):
				p = c.getEulerRotation(i, eulOrder)
				x = p[0]

				x= check360(x,last)
				last = x
				
				fobj.write(' '+'%d'%i+' '+'%f'%x)
				
			fobj.write(';\n')
			
			# y rotation
			fobj.write('\ncreateNode animCurveTA -n "'+c.getName()+'_rotY";\n')
			fobj.write('\tsetAttr -s '+'%d'%numFrames+' ".ktv['+'%d'%inPoint+':'+'%d'%outPoint+']"')
			
			last = 0.0
			for i in range(inPoint, outPoint+1):
				p = c.getEulerRotation(i, eulOrder)
				y = p[1]

				y= check360(y,last)
				last = y
				
				fobj.write(' '+'%d'%i+' '+'%f'%y)
				
			fobj.write(';\n')
			
			# z rotation
			fobj.write('\ncreateNode animCurveTA -n "'+c.getName()+'_rotZ";\n')
			fobj.write('\tsetAttr -s '+'%d'%numFrames+' ".ktv['+'%d'%inPoint+':'+'%d'%outPoint+']"')
			
			last = 0.0
			for i in range(inPoint, outPoint+1):
				p = c.getEulerRotation(i, eulOrder)
				z = -p[2]

				z= check360(z,last)
				last = z
				
				fobj.write(' '+'%d'%i+' '+'%f'%z)
				
			fobj.write(';\n')
				
		# end if (c.getExport()):
		
		idx += 1
		
	# end while (idx < numCameras):
	
	# how many groups?
	numGroups = pfpy.getNumGroups()

	for ng in range(0, numGroups):
	
		g = pfpy.getGroupRef(ng)
		
		# how many features in this group?
		numTracker = g.getNumTrackers()
		
		# how many frames in this group?
		gInPoint = g.getInPoint()
		gOutPoint = g.getOutPoint()
		gNumFrames = gOutPoint - gInPoint + 1
		
		if numTracker > 0:
			fobj.write('\ncreateNode transform -n "'+g.getName()+'" -p "pf_data";\n')
			fobj.write('\tsetAttr ".ro" 2;\n')

			# find a suitable feature size for this group
			size = g.getNullSize()
			
			for i in range(0, numTracker):
				t = g.getTrackerRef(i)
				
				if t.getSolved() and t.getExport():

					if t.getMocap() :
						# dump mocap path
						fobj.write('\ncreateNode transform -n "'+t.getName()+'" -p "'+g.getName()+'";\n')
						fobj.write('\tsetAttr ".s" -type "double3" '+'%g'%size+' '+'%g'%size+' '+'%g'%size+';\n')
					
						fobj.write('createNode locator -n "'+t.getName()+'Shape" -p "'+t.getName()+'";\n')
						fobj.write('\tsetAttr -k off ".v";\n')
						fobj.write('\tsetAttr ".uoc" yes;\n')
						fobj.write('\tsetAttr ".oc" '+'%d'%(ng%8)+';\n')

						tInPoint= t.getMocapInPoint()
						tOutPoint= t.getMocapOutPoint()
						tNumFrames= 1+tOutPoint-tInPoint

						# x position
						fobj.write('createNode animCurveTL -n "'+t.getName()+'_transX";\n')
						fobj.write('\tsetAttr -s '+'%d'%tNumFrames+' ".ktv['+'%d'%tInPoint+':'+'%d'%tOutPoint+']"')
						for j in range(tInPoint, tOutPoint+1):
							p= t.getSolvedPos(j)
							fobj.write(' '+'%d'%j+' '+'%g'%p[0])
						fobj.write(';\n')

						# y position
						fobj.write('createNode animCurveTL -n "'+t.getName()+'_transY";\n')
						fobj.write('\tsetAttr -s '+'%d'%tNumFrames+' ".ktv['+'%d'%tInPoint+':'+'%d'%tOutPoint+']"')
						for j in range(tInPoint, tOutPoint+1):
							p= t.getSolvedPos(j)
							fobj.write(' '+'%d'%j+' '+'%g'%p[1])
						fobj.write(';\n')

						# z position
						fobj.write('createNode animCurveTL -n "'+t.getName()+'_transZ";\n')
						fobj.write('\tsetAttr -s '+'%d'%tNumFrames+' ".ktv['+'%d'%tInPoint+':'+'%d'%tOutPoint+']"')
						for j in range(tInPoint, tOutPoint+1):
							p= t.getSolvedPos(j)
							fobj.write(' '+'%d'%j+' '+'%g'%p[2])
						fobj.write(';\n')

						fobj.write('connectAttr "'+t.getName()+'_transX.o" "'+t.getName()+'.tx";\n')
						fobj.write('connectAttr "'+t.getName()+'_transY.o" "'+t.getName()+'.ty";\n')
						fobj.write('connectAttr "'+t.getName()+'_transZ.o" "'+t.getName()+'.tz";\n')

					else :
						p = t.getSolvedPos()

						fobj.write('\ncreateNode transform -n "'+t.getName()+'" -p "'+g.getName()+'";\n')
						fobj.write('\tsetAttr ".s" -type "double3" '+'%g'%size+' '+'%g'%size+' '+'%g'%size+';\n')
						fobj.write('\tsetAttr ".t" -type "double3" '+'%f'%p[0]+' '+'%f'%p[1]+' '+'%f'%p[2]+';\n')
					
						fobj.write('createNode locator -n "'+t.getName()+'Shape" -p "'+t.getName()+'";\n')
						fobj.write('\tsetAttr -k off ".v";\n')
						fobj.write('\tsetAttr ".uoc" yes;\n')
						fobj.write('\tsetAttr ".oc" '+'%d'%(ng%8)+';\n')
				
					
			# is this a non-camera group?
			if not g.isCameraGroup():
				
				# x translation
				fobj.write('\ncreateNode animCurveTL -n "'+g.getName()+'_transX";\n')
				fobj.write('\tsetAttr -s '+'%d'%gNumFrames+' ".ktv['+'%d'%gInPoint+':'+'%d'%gOutPoint+']"')
				for i in range(gInPoint, gOutPoint+1):
					t = g.getTranslation(i)
					fobj.write(' '+'%d'%i+' '+'%f'%t[0])
				fobj.write(';\n')
				
				# y translation
				fobj.write('\ncreateNode animCurveTL -n "'+g.getName()+'_transY";\n')
				fobj.write('\tsetAttr -s '+'%d'%gNumFrames+' ".ktv['+'%d'%gInPoint+':'+'%d'%gOutPoint+']"')
				for i in range(gInPoint, gOutPoint+1):
					t = g.getTranslation(i)
					fobj.write(' '+'%d'%i+' '+'%f'%t[1])
				fobj.write(';\n')
				
				# z translation
				fobj.write('\ncreateNode animCurveTL -n "'+g.getName()+'_transZ";\n')
				fobj.write('\tsetAttr -s '+'%d'%gNumFrames+' ".ktv['+'%d'%gInPoint+':'+'%d'%gOutPoint+']"')
				for i in range(gInPoint, gOutPoint+1):
					t = g.getTranslation(i)
					fobj.write(' '+'%d'%i+' '+'%f'%t[2])
				fobj.write(';\n')
				
				# x rotation
				fobj.write('\ncreateNode animCurveTA -n "'+g.getName()+'_rotX";\n')
				fobj.write('\tsetAttr -s '+'%d'%gNumFrames+' ".ktv['+'%d'%gInPoint+':'+'%d'%gOutPoint+']"')
				last = 0.0
				for i in range(gInPoint, gOutPoint+1):
					r = g.getEulerRotation(i, eulOrder)
					
					# check for signs...
					x = r[0]
					if (i > gInPoint):
						x= check360(x,last)
					
					last = x
					
					fobj.write(' '+'%d'%i+' '+'%f'%x)
				
				fobj.write(';\n')
				
				# y rotation
				fobj.write('\ncreateNode animCurveTA -n "'+g.getName()+'_rotY";\n')
				fobj.write('\tsetAttr -s '+'%d'%gNumFrames+' ".ktv['+'%d'%gInPoint+':'+'%d'%gOutPoint+']"')
				last = 0.0
				for i in range(gInPoint, gOutPoint+1):
					r = g.getEulerRotation(i, eulOrder)
					
					# check for signs...
					y = r[1]
					if (i > gInPoint):
						y= check360(y,last)
					
					last = y
					
					fobj.write(' '+'%d'%i+' '+'%f'%y)

				fobj.write(';\n')
				
				# z rotation
				fobj.write('\ncreateNode animCurveTA -n "'+g.getName()+'_rotZ";\n')
				fobj.write('\tsetAttr -s '+'%d'%gNumFrames+' ".ktv['+'%d'%gInPoint+':'+'%d'%gOutPoint+']"')
				last = 0.0
				for i in range(gInPoint, gOutPoint+1):
					r = g.getEulerRotation(i, eulOrder)
					
					# check for signs...
					z = -r[2]
					if (i > gInPoint):
						z= check360(z,last)
					
					last = z
					
					fobj.write(' '+'%d'%i+' '+'%f'%z)
				
				fobj.write(';\n')

				# x scale
				fobj.write('\ncreateNode animCurveTU -n "'+g.getName()+'_scaleX";\n')
				fobj.write('\tsetAttr -s '+'%d'%gNumFrames+' ".ktv['+'%d'%gInPoint+':'+'%d'%gOutPoint+']"')
				for i in range(gInPoint, gOutPoint+1):
					s = g.getScale(i)
					fobj.write(' '+'%d'%i+' '+'%f'%s)
				
				fobj.write(';\n')

				# y scale
				fobj.write('\ncreateNode animCurveTU -n "'+g.getName()+'_scaleY";\n')
				fobj.write('\tsetAttr -s '+'%d'%gNumFrames+' ".ktv['+'%d'%gInPoint+':'+'%d'%gOutPoint+']"')
				for i in range(gInPoint, gOutPoint+1):
					s = g.getScale(i)
					fobj.write(' '+'%d'%i+' '+'%f'%s)
				
				fobj.write(';\n')

				# z scale
				fobj.write('\ncreateNode animCurveTU -n "'+g.getName()+'_scaleZ";\n')
				fobj.write('\tsetAttr -s '+'%d'%gNumFrames+' ".ktv['+'%d'%gInPoint+':'+'%d'%gOutPoint+']"')
				for i in range(gInPoint, gOutPoint+1):
					s = g.getScale(i)
					fobj.write(' '+'%d'%i+' '+'%f'%s)
				
				fobj.write(';\n')
				
				fobj.write('\nconnectAttr "'+g.getName()+'_transX.o" "'+g.getName()+'.tx";\n')
				fobj.write('connectAttr "'+g.getName()+'_transY.o" "'+g.getName()+'.ty";\n')
				fobj.write('connectAttr "'+g.getName()+'_transZ.o" "'+g.getName()+'.tz";\n')
				fobj.write('connectAttr "'+g.getName()+'_rotX.o" "'+g.getName()+'.rx";\n')
				fobj.write('connectAttr "'+g.getName()+'_rotY.o" "'+g.getName()+'.ry";\n')
				fobj.write('connectAttr "'+g.getName()+'_rotZ.o" "'+g.getName()+'.rz";\n')
				fobj.write('connectAttr "'+g.getName()+'_scaleX.o" "'+g.getName()+'.sx";\n')
				fobj.write('connectAttr "'+g.getName()+'_scaleY.o" "'+g.getName()+'.sy";\n')
				fobj.write('connectAttr "'+g.getName()+'_scaleZ.o" "'+g.getName()+'.sz";\n')

	# how many point clouds?
	numPointClouds= pfpy.getNumClouds()

	for pc in range(0, numPointClouds):
		
		c= pfpy.getCloudRef(pc)
		
		if c.getExport() :
			
			numTriangles= c.getNumTriangles()
			
			if numTriangles > 0:
				
				name= c.getName()
				
				fobj.write('\ncreateNode transform -n "'+name+'" -p "pf_data";\n')

				fobj.write('\ncreateNode mesh -n "'+name+'Shape" -p "'+name+'";\n')
				fobj.write('\tsetAttr -k off ".v";\n')
				fobj.write('\tsetAttr ".uvst[0].uvsn" -type "string" "DiffuseUV";\n')
				fobj.write('\tsetAttr ".cuvs" -type "string" "DiffuseUV";\n')
				fobj.write('\tsetAttr ".dcc" -type "string" "Ambient+Diffuse";\n')
				fobj.write('\tsetAttr ".vir" yes;\n')
				fobj.write('\tsetAttr ".vif" yes;\n')
				fobj.write('\tsetAttr ".uoc" yes;\n')
				fobj.write('\tsetAttr ".oc" '+'%d'%(modulus(pc,8))+';\n')

				# dump points
				fobj.write('\tsetAttr -s %u'%(3*numTriangles)+' ".vt[0:%u]"'%(3*numTriangles-1))
				for p in range(0,numTriangles):
					v0= c.getTrianglePoint(p,0)
					v1= c.getTrianglePoint(p,1)
					v2= c.getTrianglePoint(p,2)

					fobj.write('\n\t\t%g'%v0[0]+' %g'%v0[1]+' %g'%v0[2])
					fobj.write('\n\t\t%g'%v1[0]+' %g'%v1[1]+' %g'%v1[2])
					fobj.write('\n\t\t%g'%v2[0]+' %g'%v2[1]+' %g'%v2[2])
				fobj.write(';\n')

				# dump edges
				fobj.write('\tsetAttr -s %d'%(3*numTriangles)+' ".ed[0:%d]"'%(3*numTriangles-1))
				for t in range(0,numTriangles):
					fobj.write('\n\t\t')
					v0= 3*t+0
					v1= 3*t+1
					v2= 3*t+2
					fobj.write('%d'%v0+' %d'%v1+' 0')
					fobj.write(' %d'%v1+' %d'%v2+' 0')
					fobj.write(' %d'%v2+' %d'%v0+' 0')
				fobj.write(';\n')

				# dump triangles
				fobj.write('\tsetAttr -s %d'%numTriangles+' ".fc[0:%d]"'%(numTriangles-1)+' -type "polyFaces"')
				for t in range(0,numTriangles):
					fobj.write('\n\t\t')
					fobj.write('f 3 %d'%(3*t+0)+' %d'%(3*t+1)+' %d'%(3*t+2))
				fobj.write(';\n')
				fobj.write('\nconnectAttr "'+name+'Shape.iog" ":initialShadingGroup.dsm" -na;\n')

	# how many meshes?
	numMeshes= pfpy.getNumMeshes()

	for nm in range(0, numMeshes):

		m= pfpy.getMeshRef(nm)

		if m.getExport() :

			name= m.getName()
			numVertices= m.getNumVertices()
			numFaces= m.getNumFaces()

			fobj.write('\ncreateNode transform -n "'+name+'" -p "pf_data";\n')
			fobj.write('\tsetAttr ".ro" 2;\n')

			fobj.write('\ncreateNode mesh -n "'+name+'Shape" -p "'+name+'";\n')
			fobj.write('\tsetAttr -k off ".v";\n')
			fobj.write('\tsetAttr ".uvst[0].uvsn" -type "string" "DiffuseUV";\n')
			fobj.write('\tsetAttr ".cuvs" -type "string" "DiffuseUV";\n')
			fobj.write('\tsetAttr ".dcc" -type "string" "Ambient+Diffuse";\n')
			fobj.write('\tsetAttr ".vir" yes;\n')
			fobj.write('\tsetAttr ".vif" yes;\n')
			fobj.write('\tsetAttr ".uoc" yes;\n')
			fobj.write('\tsetAttr ".oc" '+'%d'%(modulus(nm,8))+';\n')

			# dump vertices

			fobj.write('\tsetAttr -s %d'%numVertices+' ".vt[0:%d]"'%(numVertices-1))
			for v in range(0,numVertices):
				fobj.write('\n\t\t')

				p= m.getRawVertexPos(v)
				fobj.write('%g'%p[0]+' %g'%p[1]+' %g'%p[2])
			fobj.write(';\n')
			
			totalVerts= 0
			for f in range(0,numFaces):
				totalVerts += m.getNumFaceVertices(f)
			
			# dump edges

			totalEdges= totalVerts
				
			fobj.write('\tsetAttr -s %d'%totalEdges+' ".ed[0:%d]"'%(totalEdges-1))
			for f in range(0,numFaces):
				fobj.write('\n\t\t')

				numv= m.getNumFaceVertices(f)
				for fv in range(1, numv):
					v0= m.getFaceVertex(f,fv-1)
					v1= m.getFaceVertex(f,fv)
					if fv > 1 :
						fobj.write(' ')
					fobj.write('%d'%v0+' %d'%v1+' 0')
				v0= m.getFaceVertex(f,numv-1)
				v1= m.getFaceVertex(f,0)
				fobj.write(' %d'%v0+' %d'%v1+' 0')
			fobj.write(';\n')

			if (m.hasTextureClip() or m.hasUVs()):
				# dump uv coordinates
				fobj.write('\tsetAttr ".iog[0].og[1].gcl" -type "componentList" 1 "f[0:'+'%d'%numFaces+']";\n')
				fobj.write('\tsetAttr -s %d'%(totalVerts)+' ".uvst[0].uvsp[0:%d]"'%(totalVerts-1)+' -type "float2"')
				for f in range(0,numFaces):
					fobj.write('\n\t\t')

					numv= m.getNumFaceVertices(f)
					for fv in range(0, numv):
						uv= m.getFaceVertexTexCoord(f,fv)
						if fv > 0:
							fobj.write(' ')
						fobj.write('%f'%uv[0]+' %f'%uv[1])
				fobj.write(';\n')

				# dump faces
				fobj.write('\tsetAttr -s %d'%numFaces+' ".fc[0:%d]"'%(numFaces-1)+' -type "polyFaces"')
				eidx= 0
				for f in range(0,numFaces):
					fobj.write('\n\t\t')
					numv= m.getNumFaceVertices(f)
					fobj.write('f %d'%numv)
					tmp= eidx
					for fv in range(0, numv):
						fobj.write(' %d'%(tmp))
						tmp += 1
					fobj.write('\n\t\t')
					fobj.write('mu 0 %d'%numv)
					for fv in range(0, numv):
						fobj.write(' %d'%(eidx))
						eidx += 1
				fobj.write(';\n')
				
				if not m.hasTextureClip():
					fobj.write('\nconnectAttr "'+name+'Shape.iog" ":initialShadingGroup.dsm" -na;\n')
			else:
				# dump triangles
				fobj.write('\tsetAttr -s %d'%numFaces+' ".fc[0:%d]"'%(numFaces-1)+' -type "polyFaces"')
				eidx= 0
				for f in range(0,numFaces):
					fobj.write('\n\t\t')
					numv= m.getNumFaceVertices(f)
					fobj.write('f %d'%numv)
					for fv in range(0, numv):
						fobj.write(' %d'%(eidx))
						eidx += 1
				fobj.write(';\n')
				fobj.write('\nconnectAttr "'+name+'Shape.iog" ":initialShadingGroup.dsm" -na;\n')

			# write transform
			mInPoint = m.getInPoint()
			mOutPoint = m.getOutPoint()
			mNumFrames = mOutPoint - mInPoint + 1
				
			# x translation
			fobj.write('\ncreateNode animCurveTL -n "'+name+'_transX";\n')
			fobj.write('\tsetAttr -s '+'%d'%mNumFrames+' ".ktv['+'%d'%mInPoint+':'+'%d'%mOutPoint+']"')
			for i in range(mInPoint, mOutPoint+1):
				t = m.getTranslation(i, True)
				fobj.write(' '+'%d'%i+' '+'%f'%t[0])
			fobj.write(';\n')
				
			# y translation
			fobj.write('\ncreateNode animCurveTL -n "'+name+'_transY";\n')
			fobj.write('\tsetAttr -s '+'%d'%mNumFrames+' ".ktv['+'%d'%mInPoint+':'+'%d'%mOutPoint+']"')
			for i in range(mInPoint, mOutPoint+1):
				t = m.getTranslation(i, True)
				fobj.write(' '+'%d'%i+' '+'%f'%t[1])
			fobj.write(';\n')
				
			# z translation
			fobj.write('\ncreateNode animCurveTL -n "'+name+'_transZ";\n')
			fobj.write('\tsetAttr -s '+'%d'%mNumFrames+' ".ktv['+'%d'%mInPoint+':'+'%d'%mOutPoint+']"')
			for i in range(mInPoint, mOutPoint+1):
				t = m.getTranslation(i, True)
				fobj.write(' '+'%d'%i+' '+'%f'%t[2])
			fobj.write(';\n')
				
			# x rotation
			fobj.write('\ncreateNode animCurveTA -n "'+name+'_rotX";\n')
			fobj.write('\tsetAttr -s '+'%d'%mNumFrames+' ".ktv['+'%d'%mInPoint+':'+'%d'%mOutPoint+']"')
			last = 0.0
			for i in range(mInPoint, mOutPoint+1):
				r = m.getEulerRotation(i, eulOrder, True)
					
				# check for signs...
				x = r[0]
				if (i > mInPoint):
					x= check360(x,last)
				last = x
					
				fobj.write(' '+'%d'%i+' '+'%f'%x)				
			fobj.write(';\n')
				
			# y rotation
			fobj.write('\ncreateNode animCurveTA -n "'+name+'_rotY";\n')
			fobj.write('\tsetAttr -s '+'%d'%mNumFrames+' ".ktv['+'%d'%mInPoint+':'+'%d'%mOutPoint+']"')
			last = 0.0
			for i in range(mInPoint, mOutPoint+1):
				r = m.getEulerRotation(i, eulOrder, True)
					
				# check for signs...
				y = r[1]
				if (i > mInPoint):
					y= check360(y,last)
				last = y
					
				fobj.write(' '+'%d'%i+' '+'%f'%y)
			fobj.write(';\n')
				
			# z rotation
			fobj.write('\ncreateNode animCurveTA -n "'+name+'_rotZ";\n')
			fobj.write('\tsetAttr -s '+'%d'%mNumFrames+' ".ktv['+'%d'%mInPoint+':'+'%d'%mOutPoint+']"')
			last = 0.0
			for i in range(mInPoint, mOutPoint+1):
				r = m.getEulerRotation(i, eulOrder, True)
					
				# check for signs...
				z = -r[2]
				if (i > mInPoint):
					z= check360(z,last)
				last = z
					
				fobj.write(' '+'%d'%i+' '+'%f'%z)
			fobj.write(';\n')

			# x scale
			fobj.write('\ncreateNode animCurveTU -n "'+name+'_scaleX";\n')
			fobj.write('\tsetAttr -s '+'%d'%mNumFrames+' ".ktv['+'%d'%mInPoint+':'+'%d'%mOutPoint+']"')
			for i in range(mInPoint, mOutPoint+1):
				s = m.getScale(i)
				fobj.write(' '+'%d'%i+' '+'%f'%s)
			fobj.write(';\n')

			# y scale
			fobj.write('\ncreateNode animCurveTU -n "'+name+'_scaleY";\n')
			fobj.write('\tsetAttr -s '+'%d'%mNumFrames+' ".ktv['+'%d'%mInPoint+':'+'%d'%mOutPoint+']"')
			for i in range(mInPoint, mOutPoint+1):
				s = m.getScale(i)
				fobj.write(' '+'%d'%i+' '+'%f'%s)
			fobj.write(';\n')

			# z scale
			fobj.write('\ncreateNode animCurveTU -n "'+name+'_scaleZ";\n')
			fobj.write('\tsetAttr -s '+'%d'%mNumFrames+' ".ktv['+'%d'%mInPoint+':'+'%d'%mOutPoint+']"')
			for i in range(mInPoint, mOutPoint+1):
				s = m.getScale(i)
				fobj.write(' '+'%d'%i+' '+'%f'%s)
			fobj.write(';\n')
				
			fobj.write('\nconnectAttr "'+name+'_transX.o" "'+name+'.tx";\n')
			fobj.write('connectAttr "'+name+'_transY.o" "'+name+'.ty";\n')
			fobj.write('connectAttr "'+name+'_transZ.o" "'+name+'.tz";\n')
			fobj.write('connectAttr "'+name+'_rotX.o" "'+name+'.rx";\n')
			fobj.write('connectAttr "'+name+'_rotY.o" "'+name+'.ry";\n')
			fobj.write('connectAttr "'+name+'_rotZ.o" "'+name+'.rz";\n')
			fobj.write('connectAttr "'+name+'_scaleX.o" "'+name+'.sx";\n')
			fobj.write('connectAttr "'+name+'_scaleY.o" "'+name+'.sy";\n')
			fobj.write('connectAttr "'+name+'_scaleZ.o" "'+name+'.sz";\n')

			if (m.hasTextureClip()):
				fobj.write('\ncreateNode phong -n "'+name+'PFMat";\n')
				fobj.write('\tsetAttr ".dc" 1;\n')
				fobj.write('\tsetAttr ".ambc" -type "float3" 0.2 0.2 0.2 ;\n')
				fobj.write('\tsetAttr ".sc" -type "float3" 0.2 0.2 0.2 ;\n')
				fobj.write('\tsetAttr ".rfl" 1;\n')
				fobj.write('createNode shadingEngine -n "'+name+'PFSG";\n')
				fobj.write('\tsetAttr ".ihi" 0;\n')
				fobj.write('\tsetAttr ".ro" yes;\n')
				fobj.write('createNode materialInfo -n "'+name+'PFMaterialInfo";\n')
				fobj.write('createNode groupId -n "'+name+'PFGroupId";\n')
				fobj.write('\tsetAttr ".ihi" 0;\n')
				fobj.write('createNode file -n "'+name+'PFDiffuseTexture";\n')
				fobj.write('\tsetAttr ".ftn" -type "string" "'+m.getTextureClipFilename('all', True)+'";\n')

				tinp= m.getTextureClipInPoint()
				toutp= m.getTextureClipOutPoint()

				# animated texture?
				if toutp > tinp:
					fobj.write('\tsetAttr ".ufe" yes;\n')
					fobj.write('\tsetAttr ".sce" '+'%d'%tinp+';\n')
					fobj.write('\tsetAttr ".ece" '+'%d'%toutp+';\n')

				fobj.write('createNode place2dTexture -n "'+name+'PFPlace2dTexture";\n')

				fobj.write('\nconnectAttr "'+name+'PFGroupId.id" "'+name+'Shape.iog.og[1].gid";\n')
				fobj.write('connectAttr "'+name+'PFSG.mwc" "'+name+'Shape.iog.og[1].gco";\n')
				fobj.write('connectAttr "'+name+'PFDiffuseTexture.oc" "'+name+'PFMat.c";\n')
				fobj.write('connectAttr "'+name+'PFMat.oc" "'+name+'PFSG.ss";\n')
				fobj.write('connectAttr "'+name+'PFGroupId.msg" "'+name+'PFSG.gn" -na;\n')
				fobj.write('connectAttr "'+name+'Shape.iog.og[1]" "'+name+'PFSG.dsm" -na;\n')
				fobj.write('connectAttr "'+name+'PFSG.msg" "'+name+'PFMaterialInfo.sg";\n')
				fobj.write('connectAttr "'+name+'PFMat.msg" "'+name+'PFMaterialInfo.m";\n')
				fobj.write('connectAttr "'+name+'PFDiffuseTexture.msg" "'+name+'PFMaterialInfo.t" -na;\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.o" "'+name+'PFDiffuseTexture.uv";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.ofu" "'+name+'PFDiffuseTexture.ofu";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.ofv" "'+name+'PFDiffuseTexture.ofv";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.rf" "'+name+'PFDiffuseTexture.rf";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.reu" "'+name+'PFDiffuseTexture.reu";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.rev" "'+name+'PFDiffuseTexture.rev";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.vt1" "'+name+'PFDiffuseTexture.vt1";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.vt2" "'+name+'PFDiffuseTexture.vt2";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.vt3" "'+name+'PFDiffuseTexture.vt3";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.vc1" "'+name+'PFDiffuseTexture.vc1";\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.ofs" "'+name+'PFDiffuseTexture.fs";\n')
				fobj.write('connectAttr "'+name+'PFSG.pa" ":renderPartition.st" -na;\n')
				fobj.write('connectAttr "'+name+'PFMat.msg" ":defaultShaderList1.s" -na;\n')
				fobj.write('connectAttr "'+name+'PFDiffuseTexture.msg" ":defaultTextureList1.tx" -na;\n')
				fobj.write('connectAttr "'+name+'PFPlace2dTexture.msg" ":defaultRenderUtilityList1.u" -na;\n')

				# animated texture?
				if toutp > tinp:
					fobj.write('createNode expression -n "'+name+'PFExpression";\n')
					fobj.write('\tsetAttr -k on ".nds";')
					fobj.write('\tsetAttr ".ixp" -type "string" ".O[0]=frame";')
					fobj.write('connectAttr "'+name+'PFExpression.out[0]" "'+name+'PFDiffuseTexture.fe";\n')

	if hasFirstCamera:

                pinp= firstCamera.getInPoint()
                poutp= firstCamera.getOutPoint()

                if poutp == pinp:
                        poutp= pinp+1

                fobj.write('\ncreateNode script -n "animationScriptNode";\n')
                fobj.write('\tsetAttr ".b" -type "string" "playbackOptions -min '+'%d'%pinp+' -max '+'%d'%poutp+' -ast '+'%d'%pinp+' -aet '+'%d'%poutp+'; currentTime -e '+'%d'%pinp+'";\n')

                fobj.write('\tsetAttr ".st" 6;\n')
		
	for i in range(0, numCameras):
		c = pfpy.getCameraRef(i)

		if c.getExport():

			cl = pfpy.getClipRef(i)

			# don't export if the clip is a single frame.. Maya doesn't like loading a large
			# number of image planes in at the same time so this can prevent it crashing when using
			# lots of single-frame cameras
			
			if not cl.isSingleFrame():

				fobj.write('\ncreateNode transform -n "'+c.getName()+'PlaneTransform" -p "'+c.getName()+'Shape";\n')
				fobj.write('\ncreateNode imagePlane -n "'+c.getName()+'Plane" -p "'+c.getName()+'PlaneTransform";\n')
				fobj.write('\tsetAttr -k off ".v";\n')
				fobj.write('\tsetAttr ".fc" 102;\n')
		
				xsize = c.getFrameWidth()
				ysize = c.getFrameHeight()

				if cl.isSingleFrame() or cl.isMovieFile():
					# store the entire filename and don't use an image sequence
					fobj.write('\tsetAttr ".imn" -type "string" "'+cl.getFilename('all', c.getInPoint(), True)+'";\n')

					if cl.isMovieFile():
						fobj.write('\tsetAttr ".t" 2;\n')
				else:
					# construct file path for the first frame and use an image sequence
					num= '%d'%c.getInPoint()
					num= num.zfill(cl.getFilenamePadding())
					fobj.write('\tsetAttr ".imn" -type "string" "'+cl.getFilename('dir', -1, True)+cl.getFilename('file', -1, True)+cl.getFilename('sep')+num+'.'+cl.getFilename('ext')+'";\n')

				if c.getInPoint() != c.getOutPoint():
					fobj.write('\tsetAttr ".ufe" yes;\n')

				fobj.write('\tsetAttr ".cov" -type "short2" '+'%d'%xsize+' '+'%d'%ysize+';\n')
				fobj.write('\tsetAttr ".f" 4;\n')
				fobj.write('\tsetAttr ".dic" yes;\n')
				fobj.write('\tsetAttr ".d" 1000;\n')
				fobj.write('\tsetAttr ".s" -type "double2" '+'%g'%(c.getSensorWidth('mm')/25.4)+' '+'%g'%(c.getSensorHeight('mm')/25.4)+';\n')
		
				if c.getInPoint() != c.getOutPoint():
					fobj.write('\ncreateNode expression -n "'+c.getName()+'Expression";\n')
					fobj.write('\tsetAttr -k on ".nds";\n')

					if cl.isSingleFrame():
						fobj.write('\tsetAttr ".ixp" -type "string" ".O[0]=frame+'+'%d'%(cl.getFilename('frame')-c.getInPoint())+'";\n')
					else:
						fobj.write('\tsetAttr ".ixp" -type "string" ".O[0]=frame+'+'%d'%0+'";\n')
		
	fobj.write('\ncreateNode transform -s -n "persp";\n')
	fobj.write('\tsetAttr ".v" no;\n')
	fobj.write('\tsetAttr ".t" -type "double3" 32.0 24.0 32.0;\n')
	fobj.write('\tsetAttr ".r" -type "double3" -28.0 45.0 0.0;\n')
	fobj.write('createNode camera -s -n "perspShape" -p "persp";\n')
	fobj.write('\tsetAttr -k off ".v" no;\n')
	fobj.write('\tsetAttr ".rnd" no;\n')
	fobj.write('\tsetAttr ".ncp" 0.1;\n')
	fobj.write('\tsetAttr ".fcp" 10000.0;\n')
	fobj.write('\tsetAttr ".fd" 5;\n')
	fobj.write('\tsetAttr ".coi" 100;\n')
	fobj.write('\tsetAttr ".ow" 10;\n')
	fobj.write('\tsetAttr ".imn" -type "string" "persp";\n')
	fobj.write('\tsetAttr ".den" -type "string" "persp_depth";\n')	
	fobj.write('\tsetAttr ".man" -type "string" "persp_mask";\n')
	fobj.write('\tsetAttr ".hc" -type "string" "viewSet -p %camera";\n')
		
	fobj.write('\ncreateNode transform -s -n "top";\n')
	fobj.write('\tsetAttr ".v" no;\n')
	fobj.write('\tsetAttr ".t" -type "double3" 0 100 0;\n')
	fobj.write('\tsetAttr ".r" -type "double3" -90 0 0;\n')
	fobj.write('createNode camera -s -n "topShape" -p "top";\n')
	fobj.write('\tsetAttr -k off ".v" no;\n')
	fobj.write('\tsetAttr ".rnd" no;\n')
	fobj.write('\tsetAttr ".ncp" 0.1;\n')
	fobj.write('\tsetAttr ".fcp" 10000.0;\n')
	fobj.write('\tsetAttr ".fd" 5;\n')
	fobj.write('\tsetAttr ".coi" 100;\n')
	fobj.write('\tsetAttr ".ow" 30;\n')
	fobj.write('\tsetAttr ".imn" -type "string" "top";\n')
	fobj.write('\tsetAttr ".den" -type "string" "top_depth";\n')	
	fobj.write('\tsetAttr ".man" -type "string" "top_mask";\n')
	fobj.write('\tsetAttr ".hc" -type "string" "viewSet -p %camera";\n')
	fobj.write('\tsetAttr ".o" yes;\n')
		
	fobj.write('\ncreateNode transform -s -n "front";\n')
	fobj.write('\tsetAttr ".v" no;\n')
	fobj.write('\tsetAttr ".t" -type "double3" 0 0 100;\n')
	fobj.write('\tsetAttr ".r" -type "double3" 0 0 0;\n')
	fobj.write('createNode camera -s -n "frontShape" -p "front";\n')
	fobj.write('\tsetAttr -k off ".v" no;\n')
	fobj.write('\tsetAttr ".rnd" no;\n')
	fobj.write('\tsetAttr ".ncp" 0.1;\n')
	fobj.write('\tsetAttr ".fcp" 10000.0;\n')
	fobj.write('\tsetAttr ".fd" 5;\n')
	fobj.write('\tsetAttr ".coi" 100;\n')
	fobj.write('\tsetAttr ".ow" 30;\n')
	fobj.write('\tsetAttr ".imn" -type "string" "front";\n')
	fobj.write('\tsetAttr ".den" -type "string" "front_depth";\n')	
	fobj.write('\tsetAttr ".man" -type "string" "front_mask";\n')
	fobj.write('\tsetAttr ".hc" -type "string" "viewSet -p %camera";\n')
	fobj.write('\tsetAttr ".o" yes;\n')
		
	fobj.write('\ncreateNode transform -s -n "side";\n')
	fobj.write('\tsetAttr ".v" no;\n')
	fobj.write('\tsetAttr ".t" -type "double3" 100 0 0;\n')
	fobj.write('\tsetAttr ".r" -type "double3" 0 90 0;\n')
	fobj.write('createNode camera -s -n "sideShape" -p "side";\n')
	fobj.write('\tsetAttr -k off ".v" no;\n')
	fobj.write('\tsetAttr ".rnd" no;\n')
	fobj.write('\tsetAttr ".ncp" 0.1;\n')
	fobj.write('\tsetAttr ".fcp" 10000.0;\n')
	fobj.write('\tsetAttr ".fd" 5;\n')
	fobj.write('\tsetAttr ".coi" 100;\n')
	fobj.write('\tsetAttr ".ow" 30;\n')
	fobj.write('\tsetAttr ".imn" -type "string" "side";\n')
	fobj.write('\tsetAttr ".den" -type "string" "side_depth";\n')	
	fobj.write('\tsetAttr ".man" -type "string" "side_mask";\n')
	fobj.write('\tsetAttr ".hc" -type "string" "viewSet -p %camera";\n')
	fobj.write('\tsetAttr ".o" yes;\n')
		
	if hasFirstCamera:

                xsize = firstCamera.getFrameWidth()
                ysize = firstCamera.getFrameHeight()
                asp = float(firstCamera.getPixelAspect())
		
                fobj.write('\nselect -ne :defaultResolution;\n')
                fobj.write('\tsetAttr ".w" '+'%d'%xsize+';\n')
                fobj.write('\tsetAttr ".h" '+'%d'%ysize+';\n')
                fobj.write('\tsetAttr ".pa" '+'%f'%asp+';\n')
                fobj.write('\tsetAttr ".al" yes;\n')
                fobj.write('\tsetAttr ".dar" '+'%g'%(float(xsize)*asp/float(ysize))+';\n')	
		
	for i in range(0, numCameras):
		c = pfpy.getCameraRef(i)

		if c.getExport():

			fobj.write('\nconnectAttr "'+c.getName()+'_focalLength.o" "'+c.getName()+'Shape.fl";\n')
			fobj.write('connectAttr "'+c.getName()+'_transX.o" "'+c.getName()+'.tx";\n')
			fobj.write('connectAttr "'+c.getName()+'_transY.o" "'+c.getName()+'.ty";\n')
			fobj.write('connectAttr "'+c.getName()+'_transZ.o" "'+c.getName()+'.tz";\n')
			fobj.write('connectAttr "'+c.getName()+'_rotX.o" "'+c.getName()+'.rx";\n')
			fobj.write('connectAttr "'+c.getName()+'_rotY.o" "'+c.getName()+'.ry";\n')
			fobj.write('connectAttr "'+c.getName()+'_rotZ.o" "'+c.getName()+'.rz";\n')
			fobj.write('connectAttr "'+c.getName()+'Plane.msg" "'+c.getName()+'Shape.ip" -na;\n')
			
			if c.getInPoint() != c.getOutPoint():
				fobj.write('connectAttr "'+c.getName()+'Expression.out[0]" "'+c.getName()+'Plane.fe";\n')
				fobj.write('connectAttr ":time1.o" "'+c.getName()+'Expression.tim";\n')

	fobj.write('\ncreateNode displayLayerManager -n "layerManager";\n')
	fobj.write('createNode displayLayer -n "defaultLayer";\n')
	fobj.write('createNode renderLayerManager -n "renderLayerManager";\n')
	fobj.write('createNode renderLayer -n "defaultRenderLayer";\n')
	fobj.write('\tsetAttr ".g" yes;\n')
	fobj.write('connectAttr "layerManager.dli[0]" "defaultLayer.id";\n')
	fobj.write('connectAttr "renderLayerManager.rlmi[0]" "defaultRenderLayer.rlid";\n')
		
	
	fobj.close()
	
